# users/views.py

from rest_framework import viewsets, status
from rest_framework.response import Response
from rest_framework.views import APIView
from rest_framework.decorators import action, api_view, permission_classes
from rest_framework.permissions import IsAuthenticated, AllowAny, IsAdminUser # For securing views
from rest_framework.parsers import MultiPartParser, FormParser 
from django.contrib.auth import authenticate, login, logout # Django auth functions
from django.utils.translation import gettext_lazy as _
from django.middleware.csrf import get_token # Import to manually get CSRF token

from .models import User, FacultyProfile
from .serializers import UserSerializer, FacultyProfileSerializer, LoginSerializer # Import LoginSerializer
from student_management.models import Student # Import Student model
from student_management.serializers import StudentSerializer # Import StudentSerializer


class UserViewSet(viewsets.ModelViewSet):
    queryset = User.objects.all().order_by('username')
    serializer_class = UserSerializer
    permission_classes = [IsAuthenticated] # Most actions require auth
    parser_classes = (MultiPartParser, FormParser)

    @action(detail=False, methods=['get'], url_path='me')
    def get_current_user_profile(self, request):
        """Returns the profile of the currently logged-in user."""
        serializer = self.get_serializer(request.user)
        return Response(serializer.data)
    def get_object(self):
        # Override get_object to allow users to update their own profile via /api/users/me/
        if self.kwargs.get('pk') == 'me':
            return self.request.user
        return super().get_object()

    def partial_update(self, request, *args, **kwargs):
        # Allow users to update their own profile
        instance = self.get_object()
        if instance != request.user:
            return Response({'detail': _("غير مصرح لك بتعديل ملف هذا المستخدم.")}, status=status.HTTP_403_FORBIDDEN)
        
        # Use partial=True to allow updating only some fields
        serializer = self.get_serializer(instance, data=request.data, partial=True)
        serializer.is_valid(raise_exception=True)
        self.perform_update(serializer)
        return Response(serializer.data)

    def get_permissions(self):
        if self.action in ['update', 'partial_update', 'destroy']:
            if self.request.user.is_authenticated and self.request.user.is_staff:
                return [IsAuthenticated(), IsAdminUser()]
            else:
                return [IsAuthenticated()] # Allow student to update their own profile via `me` endpoint
        return super().get_permissions()


    @action(detail=False, methods=['get'], url_path='student-profile')
    def get_current_student_profile(self, request):
        """Returns the student profile linked to the currently logged-in user."""
        try:
            # Assuming User has a OneToOneField to Student named 'student_profile'
            student = request.user.student_profile
            serializer = StudentSerializer(student, context={'request': request}) # Pass context for get_status_display
            return Response(serializer.data)
        except Student.DoesNotExist:
            return Response({'detail': _("ملف الطالب غير موجود لهذا المستخدم.")}, status=status.HTTP_404_NOT_FOUND)
        except Exception as e:
            return Response({'detail': str(e)}, status=status.HTTP_500_INTERNAL_SERVER_ERROR)


class FacultyProfileViewSet(viewsets.ModelViewSet):
    queryset = FacultyProfile.objects.all().order_by('fourth_name_ar', 'first_name_ar')
    serializer_class = FacultyProfileSerializer
    permission_classes = [IsAuthenticated]


# Custom APIView for login (replaces obtain_auth_token)
class LoginView(APIView):
    permission_classes = [AllowAny] # Allow unauthenticated access to login

    def post(self, request, *args, **kwargs):
        serializer = LoginSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        username = serializer.validated_data['username']
        password = serializer.validated_data['password']

        user = authenticate(request, username=username, password=password)

        if user is not None:
            login(request, user) # This sets the session cookie
            return Response({'detail': _("تم تسجيل الدخول بنجاح."), 'username': user.username}, status=status.HTTP_200_OK)
        else:
            return Response({'detail': _("اسم المستخدم أو كلمة المرور غير صحيحة.")}, status=status.HTTP_400_BAD_REQUEST)

# Custom APIView for logout
class LogoutView(APIView):
    permission_classes = [IsAuthenticated] # Only authenticated users can logout

    def post(self, request, *args, **kwargs):
        logout(request) # This clears the session
        return Response({'detail': _("تم تسجيل الخروج بنجاح.")}, status=status.HTTP_200_OK)

# Public endpoint to trigger CSRF cookie generation
@api_view(['GET']) # Use function-based view for simplicity
@permission_classes([AllowAny]) # This endpoint is publicly accessible
def csrf_token_view(request):
    # This view's sole purpose is to trigger Django's CsrfViewMiddleware
    # to set the csrftoken cookie on the client.
    # We explicitly get the token, but just returning an empty success is enough.
    token = get_token(request)
    return Response({'detail': _("CSRF Token obtained.")}, status=status.HTTP_200_OK)