# users/serializers.py

from rest_framework import serializers
from .models import User, FacultyProfile
from university_structure.models import Department, College # Import these for related fields if needed in FacultyProfileSerializer

class UserSerializer(serializers.ModelSerializer):
    is_registrar_flag = serializers.SerializerMethodField()
    faculty_profile_id = serializers.SerializerMethodField() 
    is_finance_officer_flag = serializers.SerializerMethodField()
    avatar_url = serializers.SerializerMethodField()


    class Meta:
        model = User
        fields = ('id', 'username', 'email', 'first_name', 'last_name', 'is_staff', 'is_active', 'is_registrar_flag', 'faculty_profile_id' , 'is_finance_officer_flag', 'avatar', 'avatar_url') 
        read_only_fields = ('is_staff', 'is_active') # Prevent accidental changes via API for these fields
    def get_is_registrar_flag(self, obj):
           return hasattr(obj, 'faculty_profile') and obj.faculty_profile.is_registrar
    def get_is_finance_officer_flag(self, obj):
        return hasattr(obj, 'faculty_profile') and obj.faculty_profile.is_finance_officer

    def get_faculty_profile_id(self, obj):
          if hasattr(obj, 'faculty_profile'):
            return obj.faculty_profile.id
          return None # Return None if no faculty_profile is linked
    def get_avatar_url(self, obj):
        if obj.avatar:
            request = self.context.get('request')
            if request is not None:
                return request.build_absolute_uri(obj.avatar.url)
            return obj.avatar.url
        return None


class FacultyProfileSerializer(serializers.ModelSerializer):
    # Read-only fields to display names from related models
    user_username = serializers.CharField(source='user.username', read_only=True)
    full_name_ar = serializers.SerializerMethodField()
    full_name_en = serializers.SerializerMethodField()
    department_name_ar = serializers.CharField(source='department.name_ar', read_only=True, allow_null=True)
    college_name_ar = serializers.CharField(source='college.name_ar', read_only=True, allow_null=True)

    class Meta:
        model = FacultyProfile
        fields = '__all__'
        # You might want to exclude 'user' if you handle user creation separately,
        # or make it read_only to link existing users. For now, it's included.

    def get_full_name_ar(self, obj):
        return f"{obj.first_name_ar} {obj.second_name_ar} {obj.third_name_ar} {obj.fourth_name_ar}"

    def get_full_name_en(self, obj):
        return f"{obj.first_name_en} {obj.second_name_en} {obj.third_name_en} {obj.fourth_name_en}"
class LoginSerializer(serializers.Serializer):
    username = serializers.CharField()
    password = serializers.CharField(write_only=True)

