# users/models.py

from django.db import models
from django.contrib.auth.models import AbstractUser, Group, Permission
from django.utils.translation import gettext_lazy as _ # Add this import

class User(AbstractUser):
    """
    Custom User Model inheriting from Django's AbstractUser.
    We will use AbstractUser's built-in first_name and last_name fields.
    """
    avatar = models.ImageField(
        upload_to='avatars/',
        null=True,
        blank=True,
        verbose_name="الصورة الشخصية"
    )
    class Meta:
        verbose_name = "مستخدم"
        verbose_name_plural = "المستخدمون"

    def __str__(self):
        if self.first_name and self.last_name:
            return f"{self.first_name} {self.last_name} ({self.username})"
        return self.username

    groups = models.ManyToManyField(
        Group,
        verbose_name=('groups'),
        blank=True,
        help_text=(
            'The groups this user belongs to. A user will get all permissions '
            'granted to each of their groups.'
        ),
        related_name="custom_user_groups_set",
        related_query_name="user",
    )
    user_permissions = models.ManyToManyField(
        Permission,
        verbose_name=('user permissions'),
        blank=True,
        help_text=('Specific permissions for this user.'),
        related_name="custom_user_permissions_set",
        related_query_name="user",
    )

class FacultyProfile(models.Model):
    """
    Profile for faculty members and staff, linked to a User account.
    This holds specific details for university employees.
    """
    user = models.OneToOneField(User, on_delete=models.CASCADE, related_name='faculty_profile', verbose_name="حساب المستخدم")
    
    # Personal Details - Four name parts for Arabic and English
    first_name_ar = models.CharField(max_length=100, verbose_name="الاسم الأول (عربي)")
    second_name_ar = models.CharField(max_length=100, verbose_name="الاسم الثاني (عربي)")
    third_name_ar = models.CharField(max_length=100, verbose_name="الاسم الثالث (عربي)")
    fourth_name_ar = models.CharField(max_length=100, verbose_name="الاسم الرابع (عربي)") # Family name
    
    first_name_en = models.CharField(max_length=100, verbose_name="First Name (English)")
    second_name_en = models.CharField(max_length=100, verbose_name="Second Name (English)")
    third_name_en = models.CharField(max_length=100, verbose_name="Third Name (English)")
    fourth_name_en = models.CharField(max_length=100, verbose_name="Fourth Name (English)") # Family name

    employee_id = models.CharField(max_length=20, unique=True, verbose_name="رقم الموظف")
    is_registrar = models.BooleanField(default=False, verbose_name=_("مسجل عام"))
    is_finance_officer = models.BooleanField(default=False, verbose_name=_("مسؤول مالي"))


    # Professional Details

    department = models.ForeignKey('university_structure.Department', on_delete=models.SET_NULL, null=True, blank=True,
                                   related_name='faculty_members', verbose_name="القسم")
    college = models.ForeignKey('university_structure.College', on_delete=models.SET_NULL, null=True, blank=True,
                                related_name='faculty_members', verbose_name="الكلية")
    
    POSITION_CHOICES = [
        ('Professor', 'أستاذ'),
        ('Associate Professor', 'أستاذ مشارك'),
        ('Assistant Professor', 'أستاذ مساعد'),
        ('Lecturer', 'محاضر'),
        ('Instructor', 'معيد'),
        ('Admin Staff', 'موظف إداري'),
        ('Head of Department', 'رئيس قسم'),
        ('Dean', 'عميد كلية'),
        ('Registrar', 'مسجل عام'),
        ('Finance Officer', 'مسؤول مالي'),
        ('System Administrator', 'مسؤول نظام'),
        # Add more positions as needed
    ]
    position = models.CharField(max_length=50, choices=POSITION_CHOICES, verbose_name="الوظيفة")
    
    hire_date = models.DateField(verbose_name="تاريخ التعيين")
    contact_phone = models.CharField(max_length=20, blank=True, null=True, verbose_name="رقم الهاتف")
    contact_email = models.EmailField(unique=True, verbose_name="البريد الإلكتروني للعمل")
    
    is_active = models.BooleanField(default=True, verbose_name="نشط")
    @property
    def full_name_ar(self):
        return f"{self.first_name_ar} {self.second_name_ar} {self.third_name_ar} {self.fourth_name_ar}"

    @property
    def full_name_en(self):
        return f"{self.first_name_en} {self.second_name_en} {self.third_name_en} {self.fourth_name_en}"

    
    class Meta:
        verbose_name = "ملف هيئة تدريس/موظف"
        verbose_name_plural = "ملفات هيئة التدريس والموظفين"

    def __str__(self):
        return self.full_name_ar