# university_structure/models.py

from django.db import models
from users.models import FacultyProfile
from django.utils.translation import gettext_lazy as _

class AcademicYear(models.Model):
    """
    Represents an academic year, e.g., 2025-2026.
    """
    year = models.CharField(max_length=9, unique=True, verbose_name=_("السنة الأكاديمية"))
    start_date = models.DateField(verbose_name=_("تاريخ البدء"))
    end_date = models.DateField(verbose_name=_("تاريخ الانتهاء"))
    is_active = models.BooleanField(default=False, verbose_name=_("نشطة حالياً"))

    class Meta:
        verbose_name = _("سنة أكاديمية")
        verbose_name_plural = _("سنوات أكاديمية")
    def __str__(self):
        return self.year

class AcademicSemester(models.Model):
    """
    Represents a semester within an academic year.
    """
    academic_year = models.ForeignKey(AcademicYear, on_delete=models.CASCADE, related_name='semesters', verbose_name="السنة الأكاديمية")
    name_ar = models.CharField(max_length=50, verbose_name=_("اسم الفصل (عربي)")) # e.g., "الفصل الأول"
    name_en = models.CharField(max_length=50, verbose_name="اسم الفصل (إنجليزي)") # e.g., "Semester 1"
    semester_number = models.IntegerField(verbose_name="رقم الفصل") # e.g., 1, 2
    start_date = models.DateField(verbose_name="تاريخ البدء")
    end_date = models.DateField(verbose_name="تاريخ الانتهاء")
    registration_start_date = models.DateField(verbose_name="بداية التسجيل")
    registration_end_date = models.DateField(verbose_name="نهاية التسجيل")
    grade_entry_start_date = models.DateField(verbose_name="بداية إدخال الدرجات")
    grade_entry_end_date = models.DateField(verbose_name="نهاية إدخال الدرجات")
    is_active = models.BooleanField(default=False, verbose_name="نشط حالياً")

    class Meta:
        unique_together = ('academic_year', 'semester_number') # Ensures only one semester number per academic year
        verbose_name = _("فصل دراسي")
        verbose_name_plural = _("فصول دراسية")

    def __str__(self):
        return f"{self.academic_year.year} - {self.name_ar}"

class College(models.Model):
    """
    Represents a College or Faculty within the university.
    """
    name_ar = models.CharField(max_length=255, verbose_name="اسم الكلية (عربي)")
    name_en = models.CharField(max_length=255, verbose_name="اسم الكلية (إنجليزي)")
    code = models.CharField(max_length=20, unique=True, verbose_name="كود الكلية")
    # Link to FacultyProfile for the Dean
    dean = models.OneToOneField(FacultyProfile, on_delete=models.SET_NULL, null=True, blank=True,
                                related_name='dean_of_college', verbose_name="العميد")
    contact_info = models.TextField(blank=True, verbose_name="معلومات الاتصال")

    class Meta:
        verbose_name = "كلية"
        verbose_name_plural = "كليات"

    def __str__(self):
        return self.name_ar

class Department(models.Model):
    """
    Represents a Department within a College.
    """
    college = models.ForeignKey(College, on_delete=models.CASCADE, related_name='departments', verbose_name="الكلية")
    name_ar = models.CharField(max_length=255, verbose_name="اسم القسم (عربي)")
    name_en = models.CharField(max_length=255, verbose_name="اسم القسم (إنجليزي)")
    code = models.CharField(max_length=20, unique=True, verbose_name="كود القسم")
    # Link to FacultyProfile for the Head of Department
    head = models.OneToOneField(FacultyProfile, on_delete=models.SET_NULL, null=True, blank=True,
                                related_name='head_of_department', verbose_name="رئيس القسم")
    contact_info = models.TextField(blank=True, verbose_name="معلومات الاتصال")

    class Meta:
        unique_together = ('college', 'code') # Ensure code is unique per college
        verbose_name = "قسم"
        verbose_name_plural = "أقسام"

    def __str__(self):
        return f"{self.name_ar} ({self.college.name_ar})"

class Batch(models.Model):
    """
    Represents a student cohort admitted in a specific academic year for a department.
    """
    department = models.ForeignKey(Department, on_delete=models.CASCADE, related_name='batches', verbose_name="القسم")
    academic_year = models.ForeignKey(AcademicYear, on_delete=models.CASCADE, related_name='batches', verbose_name="السنة الأكاديمية")
    name = models.CharField(max_length=100, verbose_name="اسم الدفعة") # e.g., "Computer Science Batch 2025"
    start_year = models.IntegerField(verbose_name="سنة البدء")
    end_year = models.IntegerField(verbose_name="سنة الانتهاء المتوقعة", null=True, blank=True)
    is_active = models.BooleanField(default=True, verbose_name="نشطة")

    class Meta:
        unique_together = ('department', 'academic_year') # A department can have only one batch per academic year
        verbose_name = "دفعة طلابية"
        verbose_name_plural = "دفعات طلابية"

    def __str__(self):
        return f"{self.name} - {self.academic_year.year}"

class UniversityConfiguration(models.Model):
    """
    Global configuration settings for the university.
    Designed to be a singleton (only one instance).
    """
    name = models.CharField(max_length=255, verbose_name="اسم الجامعة")
    current_academic_year = models.OneToOneField(
        AcademicYear,
        on_delete=models.SET_NULL,
        null=True, blank=True,
        related_name='current_university_config',
        verbose_name="السنة الأكاديمية الحالية"
    )
    current_semester = models.OneToOneField(
        AcademicSemester,
        on_delete=models.SET_NULL,
        null=True, blank=True,
        related_name='current_university_config',
        verbose_name="الفصل الدراسي الحالي"
    )
    # start_date and end_date are already covered by AcademicYear, so not strictly needed here unless for overall university operational dates.
    # For now, let's assume current_academic_year covers these periods.

    class Meta:
        verbose_name = "إعدادات الجامعة"
        verbose_name_plural = "إعدادات الجامعة"

    def __str__(self):
        return f"إعدادات {self.name}"

    def save(self, *args, **kwargs):
        # Ensure only one instance of UniversityConfiguration exists
        if UniversityConfiguration.objects.exists() and not self.pk:
            raise ValueError("There can be only one UniversityConfiguration instance")
        super().save(*args, **kwargs)