from django.shortcuts import render


# student_management/views.py

from rest_framework import viewsets, status
from rest_framework.response import Response
from django.conf import settings
from rest_framework.views import APIView
from rest_framework.parsers import MultiPartParser, FormParser # For file uploads
from rest_framework.permissions import IsAuthenticated, AllowAny, IsAdminUser # Import IsAuthenticated for permissions
from rest_framework.decorators import action # <-- Import action decorator
from django.db import transaction, models
from django.utils import timezone
from django.shortcuts import get_object_or_404
from django.utils.translation import gettext_lazy as _
from django.contrib.auth import get_user_model # To get custom User model
from django.core.mail import send_mail # For sending emails
import pandas as pd # For reading Excel/CSV
from django_filters.rest_framework import DjangoFilterBackend # <--- ADD THIS IMPORT
from rest_framework import filters
import traceback
import random



from .models import Student, SemesterEnrollment, AdmissionDecision
from .models import Application  # Add this import for Application model
from .serializers import StudentSerializer, SemesterEnrollmentSerializer, StudentRegistrationSerializer, AcceptedStudentsImportSerializer, RegistrarApprovalSerializer ,DepartmentSerializer 
from .serializers import ApplicationSerializer, AdmissionDecisionSerializer
from university_structure.models import AcademicYear, AcademicSemester, Department, Batch
from course_management.models import ProgramCourse, CourseOffering
from examination_management.models import OverallResult # For academic standing check
from finance_management.models import StudentInvoice, InvoiceItem, FeeType, FeeStructure, StudentPayment
from users.models import FacultyProfile # Import FacultyProfile to check is_registrar

import pandas as pd # For reading Excel/CSV

User = get_user_model() # Get your custom User model
class StudentViewSet(viewsets.ModelViewSet):
    # For list view, keep it light for performance.
    # For retrieve (detail view), make it comprehensive.
    queryset = Student.objects.all().select_related(
        'department', 'batch', 'current_academic_year', 'current_academic_semester'
    ).prefetch_related(
        'invoices__payments', 'invoices__items'
    ).order_by('fourth_name_ar', 'first_name_ar')
    
    serializer_class = StudentSerializer
    permission_classes = [IsAuthenticated]

    filter_backends = [DjangoFilterBackend, filters.SearchFilter]
    filterset_fields = ['department', 'batch', 'status', 'academic_level', 'current_academic_year', 'current_academic_semester']
    search_fields = ['student_id', 'first_name_ar', 'fourth_name_ar', 'contact_email', 'national_id', 'contact_phone']

    def get_permissions(self):
        if self.action in ['update', 'partial_update', 'destroy']:
            if self.request.user.is_authenticated and self.request.user.is_staff:
                return [IsAuthenticated(), IsAdminUser()]
            else:
                return [IsAuthenticated()]
        return super().get_permissions()
##########################################################
  
        return super().get_queryset() # Use default queryset for list view

    def update(self, request, *args, **kwargs):
        try:
            instance = self.get_object()
            # For update, ensure we only allow updating fields that should be editable
            # The StudentSerializer with fields = '__all__' will try to update everything.
            # If you want to restrict what can be updated via this endpoint,
            # you would define a separate StudentUpdateSerializer with limited fields.
            serializer = self.get_serializer(instance, data=request.data, partial=True)
            serializer.is_valid(raise_exception=True)
            self.perform_update(serializer)
            return Response(serializer.data)
        except Exception as e:
            traceback.print_exc()
            return Response({'detail': str(e)}, status=status.HTTP_400_BAD_REQUEST)

    def destroy(self, request, *args, **kwargs):
        try:
            instance = self.get_object()
            self.perform_destroy(instance)
            return Response(status=status.HTTP_204_NO_CONTENT)
        except Exception as e:
            traceback.print_exc()
            return Response({'detail': str(e)}, status=status.HTTP_400_BAD_REQUEST)

    @action(detail=True, methods=['put'], url_path='complete-registration', parser_classes=(MultiPartParser, FormParser), permission_classes=[IsAuthenticated])
    def complete_registration(self, request, pk=None):
        student = self.get_object()
        
        if student.user != request.user:
            return Response({'detail': _("غير مصرح لك بتعديل ملف هذا الطالب.")}, status=status.HTTP_403_FORBIDDEN)

        if student.status != 'Provisional':
            return Response({'detail': _("لا يمكن إكمال التسجيل إلا للطلاب المقبولين مبدئياً.")}, status=status.HTTP_400_BAD_REQUEST)

        try:
            # Update personal details from request.data
            student.date_of_birth = request.data.get('date_of_birth', student.date_of_birth)
            student.gender = request.data.get('gender', student.gender)
            student.nationality = request.data.get('nationality', student.nationality)
            student.contact_phone = request.data.get('contact_phone', student.contact_phone)
            student.address = request.data.get('address', student.address)

            # Handle file uploads (logic to save files)
            # IMPORTANT: For FileField, you need to save the file object, not just set boolean
            # This requires MEDIA_ROOT and MEDIA_URL to be set in settings.py
            if 'id_card_file' in request.FILES:
                student.id_card_uploaded = True
                # Example: student.id_card_file = request.FILES['id_card_file']
                # student.save() # Save the file field
            if 'high_school_cert_file' in request.FILES:
                student.high_school_cert_uploaded = True
            if 'personal_photo_file' in request.FILES:
                student.personal_photo_uploaded = True

            student.save() # Save all changes to student model

            return Response({'detail': _("تم استكمال بيانات التسجيل والمستندات بنجاح. يرجى انتظار مراجعة مسؤول القبول."), 'student_profile': StudentSerializer(student, context={'request':request}).data}, status=status.HTTP_200_OK)
        except Exception as e:
            traceback.print_exc()
            return Response({'detail': str(e)}, status=status.HTTP_400_BAD_REQUEST)
# New API for public application submission
class PublicApplicationAPIView(APIView):
    permission_classes = [AllowAny]

    def post(self, request, *args, **kwargs):
        serializer = ApplicationSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        
        try:
            with transaction.atomic():
                application = serializer.save()
                
                application_fee_type = get_object_or_404(FeeType, name_en='Application Fee')
                
                current_academic_year = AcademicYear.objects.get(is_active=True)
                current_academic_semester = AcademicSemester.objects.get(is_active=True, academic_year=current_academic_year)
                
                invoice_number = f"APP-{application.id}-{random.randint(100, 999)}"
                
                invoice = StudentInvoice.objects.create(
                    student=application,
                    academic_year=current_academic_year,
                    academic_semester=current_academic_semester,
                    invoice_number=invoice_number,
                    total_amount=application_fee_type.amount,
                    due_date=current_academic_semester.registration_end_date,
                    status='Pending'
                )
                
                InvoiceItem.objects.create(
                    invoice=invoice,
                    fee_type=application_fee_type,
                    amount=application_fee_type.amount,
                    description_ar='رسوم تقديم'
                )
            
            return Response({
                'detail': _("تم استلام طلب التقديم بنجاح. سيتم توجيهك الآن لصفحة الدفع."),
                'application_id': application.id,
                'redirect_url': f'/pay-application-fee/?invoice_id={invoice.id}'
            }, status=status.HTTP_201_CREATED)

        except FeeType.DoesNotExist:
            return Response({'detail': _("لم يتم العثور على نوع رسوم التقديم.")}, status=status.HTTP_404_NOT_FOUND)
        except Exception as e:
            traceback.print_exc()
            return Response({'detail': str(e)}, status=status.HTTP_500_INTERNAL_SERVER_ERROR)

class ApplicationViewSet(viewsets.ModelViewSet):
    queryset = Application.objects.all().order_by('-application_date')
    serializer_class = ApplicationSerializer
    permission_classes = [IsAuthenticated, IsAdminUser]
    filter_backends = [DjangoFilterBackend, filters.SearchFilter]
    filterset_fields = ['application_status', 'desired_department']
    search_fields = ['first_name_ar', 'fourth_name_ar', 'national_id', 'contact_email']


class AdmissionDecisionViewSet(viewsets.ModelViewSet):
    queryset = AdmissionDecision.objects.all().order_by('-decision_date')
    serializer_class = AdmissionDecisionSerializer
    # Add filters, e.g., for decision type or related application

class SemesterEnrollmentViewSet(viewsets.ModelViewSet):
    queryset = SemesterEnrollment.objects.all().order_by('-academic_year__year', '-academic_semester__semester_number', 'student__fourth_name_ar')
    serializer_class = SemesterEnrollmentSerializer
    # Add filters, e.g., for specific academic_year, semester, or student
class StudentRegistrationAPIView(APIView):
    """
    API endpoint for student semester registration.
    Handles academic standing check, payment simulation, and automatic course enrollment.
    """
    def post(self, request, *args, **kwargs):
        serializer = StudentRegistrationSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        student_id = serializer.validated_data['student_id']
        academic_year_id = serializer.validated_data['academic_year_id']
        academic_semester_id = serializer.validated_data['academic_semester_id']
        payment_verified = serializer.validated_data['payment_verified']

        try:
            student = get_object_or_404(Student, student_id=student_id)
            academic_year = get_object_or_404(AcademicYear, id=academic_year_id)
            academic_semester = get_object_or_404(AcademicSemester, id=academic_semester_id)
        except Exception as e:
            return Response({'detail': _("بيانات الطالب أو السنة/الفصل الأكاديمي غير صحيحة.")}, status=status.HTTP_400_BAD_REQUEST)

        # --- 1. Academic Standing Check (Crucial for progression) ---
        # Get the previous academic year. Assuming academic_year.year is an integer or can be converted.
        # This logic needs to be robust for the first year student.
        previous_academic_year = None
        try:
            previous_year_number = int(academic_year.year) - 1 # Assuming year is like '2024-2025'
            # Find the AcademicYear object for the previous year
            previous_academic_year = AcademicYear.objects.get(year__startswith=str(previous_year_number))
        except (ValueError, AcademicYear.DoesNotExist):
            # This is likely a first-year student or no previous year data exists.
            # For first-year students, no previous result check is needed.
            # For others, if previous_academic_year is None, it means no prior year record was found.
            pass

        if previous_academic_year:
            # Check overall result for the previous academic year
            # We need to find the student's overall result for the *previous* academic year's *final* semester.
            # This logic assumes OverallResult tracks the student's cumulative status.
            # For the specific requirement "حالة نتيجته في العام السابق نجاح (ليست اعادة العام ولا مفصول)"
            # we check the student's overall academic standing at the end of the previous year.
            
            # This part needs refinement based on how 'success' is defined.
            # Assuming 'OverallResult' for the previous year (or the latest one) reflects this.
            # If 'OverallResult' is updated annually/semesterly, we need to fetch the relevant one.
            # For simplicity, let's check the latest OverallResult for the student.
            
            latest_overall_result = OverallResult.objects.filter(student=student).order_by('-id').first()
            
            if not latest_overall_result:
                # If no overall result exists, it could be a new student or data issue.
                # For now, let's allow if no previous result, but in real system, this needs careful handling.
                pass
            elif latest_overall_result.academic_standing in ['Suspended', 'Expelled', 'Repeat Year']: # Add 'Repeat Year' if it's a specific status
                return Response({'detail': _("لا يمكن للطالب التسجيل بسبب حالته الأكاديمية في العام السابق: ") + latest_overall_result.academic_standing}, status=status.HTTP_403_FORBIDDEN)
            # You might also want to check for minimum CGPA here if 'success' implies a CGPA threshold.
            # For now, we only check explicit negative statuses.

        # --- 2. Payment Verification ---
        if not payment_verified:
            # In a real system, this would trigger payment gateway or manual verification.
            # For now, if payment_verified is False, we deny registration.
            return Response({'detail': _("لم يتم التحقق من دفع الرسوم. يرجى إتمام عملية الدفع أولاً.")}, status=status.HTTP_402_PAYMENT_REQUIRED)
        
        # --- 3. Check if already registered for this semester ---
        if SemesterEnrollment.objects.filter(student=student, academic_semester=academic_semester).exists():
            return Response({'detail': _("الطالب مسجل بالفعل في هذا الفصل الدراسي.")}, status=status.HTTP_400_BAD_REQUEST)

        # --- Transaction for Atomicity ---
        with transaction.atomic():
            # Create a basic invoice and mark as paid for simulation
            # In a real system, this would be more complex, involving actual fee calculation
            # based on FeeStructure and payment gateway response.
            invoice_number = f"INV-{student.student_id}-{academic_year.year}-{academic_semester.semester_number}-{timezone.now().strftime('%Y%m%d%H%M%S')}"
            registration_fee_type = get_object_or_404(FeeType, name_en='Registration Fee') # Assuming you have this FeeType
            
            # Find applicable registration fee amount
            registration_fee_amount = 0
            try:
                # Get the most specific fee structure for registration fee for this student's context
                fee_structure = FeeStructure.objects.filter(
                    fee_type=registration_fee_type,
                    academic_year=academic_year,
                    academic_semester=academic_semester,
                    department=student.department,
                    batch=student.batch
                ).first() # Try to get specific first
                
                if not fee_structure: # If no specific, try less specific
                    fee_structure = FeeStructure.objects.filter(
                        fee_type=registration_fee_type,
                        academic_year=academic_year,
                        academic_semester=academic_semester,
                        department=student.department,
                        batch__isnull=True
                    ).first()
                if not fee_structure: # If still no, try even less specific
                    fee_structure = FeeStructure.objects.filter(
                        fee_type=registration_fee_type,
                        academic_year=academic_year,
                        academic_semester__isnull=True,
                        department__isnull=True,
                        batch__isnull=True
                    ).first()
                
                if fee_structure:
                    registration_fee_amount = fee_structure.amount
                else:
                    return Response({'detail': _("لم يتم العثور على هيكل رسوم التسجيل لهذا الفصل.")}, status=status.HTTP_400_BAD_REQUEST)

            except FeeType.DoesNotExist:
                return Response({'detail': _("نوع رسوم التسجيل غير موجود في النظام.")}, status=status.HTTP_500_INTERNAL_SERVER_ERROR)


            invoice = StudentInvoice.objects.create(
                student=student,
                academic_year=academic_year,
                academic_semester=academic_semester,
                invoice_number=invoice_number,
                total_amount=registration_fee_amount, # Only registration fee for now
                due_date=timezone.now().date() + timezone.timedelta(days=7), # Due in 7 days
                status='Paid' # Marked as paid due to payment_verified=True
            )
            InvoiceItem.objects.create(
                invoice=invoice,
                fee_type=registration_fee_type,
                amount=registration_fee_amount,
                description_ar=_("رسوم تسجيل الفصل الدراسي")
            )
            # Create a payment record for the invoice
            StudentPayment.objects.create(
                invoice=invoice,
                amount=registration_fee_amount,
                payment_method='Online', # Assuming online payment for automated registration
                reference_number=f"PAY-{student.student_id}-{timezone.now().strftime('%Y%m%d%H%M%S')}",
                # received_by= None # If no specific staff member
            )


            # --- 4. Automatic Course Enrollment ---
            # Get compulsory courses for the student's department, current academic level, and semester
            compulsory_courses = ProgramCourse.objects.filter(
                department=student.department,
                academic_year_level=student.academic_level,
                semester_number=academic_semester.semester_number # Assuming semester_number in AcademicSemester matches ProgramCourse
            ).select_related('course') # Optimize query

            if not compulsory_courses.exists():
                return Response({'detail': _("لم يتم العثور على مقررات إجبارية لهذا المستوى والفصل الدراسي.")}, status=status.HTTP_404_NOT_FOUND)

            # Enroll student in each compulsory course
            for pc in compulsory_courses:
                # Find an available CourseOffering for this course in the current semester
                # This assumes there's at least one offering for each compulsory course.
                # In a real system, you might need more complex logic to select an offering (e.g., capacity, section preference)
                course_offering = CourseOffering.objects.filter(
                    course=pc.course,
                    academic_semester=academic_semester
                ).first()

                if not course_offering:
                    # Log this or return an error if a compulsory course has no offering
                    return Response({'detail': _(f"المقرر الإجباري {pc.course.code} لا يحتوي على طرح متاح في هذا الفصل.")}, status=status.HTTP_404_NOT_FOUND)
                
                # Create SemesterEnrollment
                SemesterEnrollment.objects.create(
                    student=student,
                    course_offering=course_offering,
                    academic_year=academic_year,
                    academic_semester=academic_semester,
                    enrollment_date=timezone.now().date(),
                    status='Enrolled' # Default status
                )
            
            # --- 5. Update Student's Current Academic Status ---
            student.current_academic_year = academic_year
            student.current_academic_semester = academic_semester
            # Increment academic_level if this is the first semester of a new academic year
            # This logic needs to be carefully managed, typically done at end of academic year.
            # For now, we assume academic_level is updated by an admin action for progression.
            # student.academic_level += 1 # This should be handled by an "End of Year" process
            student.is_registered_for_current_semester = True
            student.status = 'Active' # Ensure student is active after registration
            student.save()

        return Response({'detail': _("تم تسجيل الطالب بنجاح في الفصل الدراسي والمقررات الإجبارية.")}, status=status.HTTP_200_OK)
class ImportAcceptedStudentsAPIView(APIView):
    parser_classes = (MultiPartParser, FormParser)

    def post(self, request, *args, **kwargs):
        serializer = AcceptedStudentsImportSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        file = serializer.validated_data['file']

        try:
            # --- CRITICAL CORRECTION: Use pd.read_excel if you upload .xlsx, or pd.read_csv if you upload .csv ---
            # If your Excel is .xlsx:
            df = pd.read_excel(file, engine='openpyxl')
            # If your file is .csv:
            # df = pd.read_csv(file)
            # Make sure you upload the correct file type from the frontend that matches this line.

            # Assuming columns: 'student_id', 'first_name_ar', 'fourth_name_ar', 'email', 'department_code', 'password', 'national_id' (optional)
            required_cols = ['student_id', 'first_name_ar', 'fourth_name_ar', 'email', 'department_code', 'password'] # 'password' is now required
            if not all(col in df.columns for col in required_cols):
                return Response({'detail': _("الملف لا يحتوي على جميع الأعمدة المطلوبة: ") + ", ".join(required_cols)}, status=status.HTTP_400_BAD_REQUEST)

            created_students_count = 0
            errors = []

            with transaction.atomic():
                for index, row in df.iterrows():
                    try:
                        # Extract data from row, ensure stripping whitespace and converting to string
                        student_id = str(row['student_id']).strip()
                        first_name_ar = str(row['first_name_ar']).strip()
                        fourth_name_ar = str(row['fourth_name_ar']).strip()
                        email = str(row['email']).lower().strip()
                        department_code = str(row['department_code']).strip()
                        initial_password = str(row['password']).strip() # <--- CORRECTLY GETS PASSWORD FROM FILE
                        national_id = str(row.get('national_id', '')).strip() # Optional

                        # Optional English names, phone, etc. (ensure stripping for all strings)
                        first_name_en = str(row.get('first_name_en', '')).strip()
                        second_name_ar = str(row.get('second_name_ar', '')).strip()
                        third_name_ar = str(row.get('third_name_ar', '')).strip()
                        second_name_en = str(row.get('second_name_en', '')).strip()
                        third_name_en = str(row.get('third_name_en', '')).strip()
                        fourth_name_en = str(row.get('fourth_name_en', '')).strip()
                        contact_phone = str(row.get('contact_phone', '')).strip()
                        batch_name = str(row.get('batch_name', '')).strip()

                        # Find Department and Batch
                        department = get_object_or_404(Department, code=department_code)
                        current_academic_year = AcademicYear.objects.filter(is_active=True).first()
                        batch = None
                        if batch_name and current_academic_year:
                            batch = Batch.objects.filter(department=department, academic_year=current_academic_year, name=batch_name).first()
                            if not batch:
                                errors.append(f"Row {index+2}: {_('لم يتم العثور على الدفعة')}: {batch_name}")
                                continue

                        # Check for existing student_id, email, or national_id
                        if Student.objects.filter(student_id=student_id).exists():
                            errors.append(f"Row {index+2}: {_('طالب بهذا الرقم الجامعي مسجل بالفعل')}: {student_id}")
                            continue
                        if Student.objects.filter(contact_email=email).exists():
                            errors.append(f"Row {index+2}: {_('طالب بهذا البريد الإلكتروني مسجل بالفعل')}: {email}")
                            continue
                        if national_id and Student.objects.filter(national_id=national_id).exists():
                            errors.append(f"Row {index+2}: {_('طالب بهذا الرقم الوطني مسجل بالفعل')}: {national_id}")
                            continue
                        
                        # Create User account with student_id as username and password from file
                        user = User.objects.create_user(
                            username=student_id,       # <--- USE student_id as username
                            email=email,
                            password=initial_password, # <--- USE password from file
                            first_name=first_name_en,  # Django's built-in User fields
                            last_name=fourth_name_en
                        )

                        # Create Student profile
                        student = Student.objects.create(
                            user=user,
                            student_id=student_id,
                            national_id=national_id if national_id else None,
                            first_name_ar=first_name_ar,
                            second_name_ar=second_name_ar,
                            third_name_ar=third_name_ar,
                            fourth_name_ar=fourth_name_ar,
                            first_name_en=first_name_en,
                            second_name_en=second_name_en,
                            third_name_en=third_name_en,
                            fourth_name_en=fourth_name_en,
                            contact_email=contact_phone, # <--- CHECK THIS LINE: Should be email
                            contact_phone=contact_phone,
                            address=str(row.get('address', '')).strip(), # Added address field if in file
                            date_of_birth=row.get('date_of_birth', None), # Added date_of_birth if in file
                            gender=str(row.get('gender', '')).strip(), # Added gender if in file
                            nationality=str(row.get('nationality', '')).strip(), # Added nationality if in file
                            department=department,
                            batch=batch,
                            admission_date=timezone.now().date(),
                            status='Provisional',
                            academic_level=1,
                            current_academic_year=current_academic_year,
                        )
                        # Fix: Ensure contact_email is mapped correctly from CSV.
                        # I've used contact_phone for contact_email previously by mistake.
                        # You might need to add contact_email = email to the Student.objects.create above if you want to explicitly map it from the CSV email.
                        # The code I provided earlier used email for contact_email,
                        # but then reassigned contact_phone to contact_email in Student.objects.create.
                        # Let's ensure contact_email is assigned correctly.
                        student.contact_email = email # Ensure the correct email is saved to student profile
                        student.save() # Save the change if assigned after creation


                        # Send email (update message to reflect student_id as username and password)
                        login_link = "http://localhost:5173/login" 
                        send_mail(
                            subject=_("تهانينا! تم قبولك في الجامعة"),
                            message=_("عزيزي") + f" {first_name_ar},\n\n" +
                                    _("يسعدنا أن نهنئك بقبولك في جامعتنا. يرجى استكمال عملية التسجيل عبر الرابط التالي:") +
                                    f" {login_link}\n" +
                                    _("اسم المستخدم: (الرقم الجامعي الخاص بك) ") + f"{student_id}\n" +
                                    _("كلمة المرور: ") + f"{initial_password}\n\n" + # <--- Correct message
                                    _("مع خالص تحياتنا،\nجامعة العلوم والدراسات التطبيقية"),
                            from_email=settings.DEFAULT_FROM_EMAIL, # Use the default from email from settings
                            recipient_list=[email],
                            fail_silently=False,
                        )
                        created_students_count += 1

                    except Exception as row_error:
                        # Log full traceback for better debugging of row errors
                        import traceback
                        traceback.print_exc() # Print full Python traceback to server console
                        errors.append(f"Row {index+2}: {_('خطأ في معالجة بيانات الطالب')}: {row_error} (Email: {row.get('email', 'N/A')}, Student ID: {row.get('student_id', 'N/A')})")
        except Exception as e:
            import traceback
            traceback.print_exc() # Print full Python traceback to server console for file errors
            return Response({'detail': _("خطأ في قراءة أو معالجة الملف: ") + str(e)}, status=status.HTTP_400_BAD_REQUEST)

        if errors:
            return Response({'message': _("تم استيراد بعض الطلاب بنجاح، ولكن حدثت أخطاء لبعض السجلات."), 'created_count': created_students_count, 'errors': errors}, status=status.HTTP_206_PARTIAL_CONTENT)
        
        return Response({'message': _("تم استيراد جميع الطلاب بنجاح.") + f" ({created_students_count} {_('طالب تم إنشاؤه')})"}, status=status.HTTP_200_OK)

class RegistrarActionsAPIView(APIView):
    permission_classes = [IsAuthenticated] # Only authenticated users can access

    def has_registrar_permission(self, request):
        # Check if the logged-in user has a FacultyProfile and is marked as a registrar
        return request.user.is_authenticated and hasattr(request.user, 'faculty_profile') and request.user.faculty_profile.is_registrar

    def post(self, request, *args, **kwargs):
        # This API will handle approving a provisional student to active student
        if not self.has_registrar_permission(request):
            return Response({'detail': _("ليس لديك الصلاحيات اللازمة للقيام بهذا الإجراء.")}, status=status.HTTP_403_FORBIDDEN)

        serializer = RegistrarApprovalSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        student_id_from_request = serializer.validated_data['student_id']
        documents_approved_status = serializer.validated_data['documents_approved_status']

        try:
            student = get_object_or_404(Student, student_id=student_id_from_request)
        except Exception:
            return Response({'detail': _("الطالب غير موجود.")}, status=status.HTTP_404_NOT_FOUND)

        if student.status != 'Provisional':
            return Response({'detail': _("لا يمكن اعتماد هذا الطالب. حالته ليست 'مقبول مبدئياً'.")}, status=status.HTTP_400_BAD_REQUEST)

        # Update documents_approved status
        student.documents_approved = documents_approved_status
        student.save()

        # If documents are approved, change status to Active
        if student.documents_approved:
            with transaction.atomic():
                student.status = 'Active' # Change student status to Active
                student.academic_level = 1 # Confirm initial academic level if not set
                # You might need to assign current_academic_year/semester if not done by import
                if not student.current_academic_year:
                    student.current_academic_year = AcademicYear.objects.filter(is_active=True).first()
                if not student.current_academic_semester and student.current_academic_year:
                    # Assuming a default semester (e.g., semester 1) for first registration
                    student.current_academic_semester = AcademicSemester.objects.filter(
                        academic_year=student.current_academic_year, semester_number=1
                    ).first()

                student.save()
            return Response({'detail': _("تم اعتماد مستندات الطالب وتحويل حالته إلى 'طالب حالي' بنجاح."), 'student_profile': StudentSerializer(student).data}, status=status.HTTP_200_OK)
        else:
            return Response({'detail': _("تم تحديث حالة اعتماد مستندات الطالب إلى 'غير معتمدة'. الطالب ما زال 'مقبول مبدئياً'."), 'student_profile': StudentSerializer(student).data}, status=status.HTTP_200_OK)
class StudentSearchAPIView(APIView):
    permission_classes = [IsAuthenticated, IsAdminUser]
    
    def get(self, request, *args, **kwargs):
        query = request.query_params.get('query', '')
        if not query:
            return Response({'detail': _("الرجاء تقديم كلمة بحث.")}, status=status.HTTP_400_BAD_REQUEST)

        students = Student.objects.filter(
            models.Q(student_id__icontains=query) |
            models.Q(first_name_ar__icontains=query) |
            models.Q(fourth_name_ar__icontains=query) |
            models.Q(contact_email__icontains=query)
        )[:10] # Limit results for performance

        serializer = StudentSerializer(students, many=True)
        return Response(serializer.data)
class PublicDepartmentsAPIView(APIView):
    permission_classes = [AllowAny]
    
    def get(self, request, *args, **kwargs):
        departments = Department.objects.all().order_by('name_ar')
        serializer = DepartmentSerializer(departments, many=True)
        return Response(serializer.data, status=status.HTTP_200_OK)



class PublicApplicationAPIView(APIView):
    permission_classes = [AllowAny]
    parser_classes = (MultiPartParser, FormParser) # <--- CRITICAL FIX: Allow FormData

    def post(self, request, *args, **kwargs):
        serializer = ApplicationSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        
        try:
            with transaction.atomic():
                application = serializer.save()
                
                # Check for an existing 'Application Fee' type
                # The logic for creating a student invoice will now be handled separately after acceptance
                
            return Response({
                'detail': _("تم استلام طلب التقديم بنجاح. سيتم مراجعته."),
                'application_id': application.id,
            }, status=status.HTTP_201_CREATED)

        except Exception as e:
            traceback.print_exc()
            return Response({'detail': str(e)}, status=status.HTTP_500_INTERNAL_SERVER_ERROR)