# student_management/serializers.py
from django.utils.translation import gettext_lazy as _ 
from rest_framework import serializers
from .models import Student, Application, AdmissionDecision, SemesterEnrollment
from university_structure.models import Department, Batch, AcademicYear, AcademicSemester
from finance_management.serializers import StudentInvoiceSerializer, StudentPaymentSerializer # <--- Import Financial Serializers
from examination_management.serializers import SemesterResultSerializer, OverallResultSerializer, StudentCourseResultSerializer # <--- Import Exam Serializers
from university_structure.serializers import AcademicYearSerializer, AcademicSemesterSerializer, DepartmentSerializer, BatchSerializer # <--- Import Structure Serializers
from django.db.models import Sum

class DepartmentSerializer(serializers.ModelSerializer):
    class Meta:
        model = Department
        fields = ('id', 'name_ar', 'name_en')

class BatchSerializer(serializers.ModelSerializer):
    class Meta:
        model = Batch
        fields = ('id', 'name', 'academic_year_name')

class StudentSerializer(serializers.ModelSerializer):
    full_name_ar = serializers.ReadOnlyField()
    full_name_en = serializers.ReadOnlyField()
    # Added fields for department and batch
    department_name_ar = serializers.CharField(source='department.name_ar', read_only=True, allow_null=True)
    department_name_en = serializers.CharField(source='department.name_en', read_only=True, allow_null=True)
    batch_name = serializers.CharField(source='batch.name', read_only=True, allow_null=True)
    status_display = serializers.CharField(source='get_status_display', read_only=True)

    # New fields for financial summary
    total_fees = serializers.SerializerMethodField()
    total_paid = serializers.SerializerMethodField()
    total_remaining = serializers.SerializerMethodField()
    
    # Nested Financial Data (Invoices and Payments related to this student)
    invoices = StudentInvoiceSerializer(many=True, read_only=True) # related_name='invoices' on StudentInvoice
    payments = StudentPaymentSerializer(many=True, read_only=True) # related_name='payments' on StudentPayment

    # Nested Academic Results Data
    semester_results = SemesterResultSerializer(many=True, read_only=True) # related_name='semester_results' on SemesterResult
    overall_result = OverallResultSerializer(read_only=True) # related_name='overall_result' on OverallResult
    course_results = StudentCourseResultSerializer(many=True, read_only=True) # related_name='course_results' on StudentCourseResult


    class Meta:
        model = Student
        fields = '__all__' # Ensure all fields are included for comprehensive management

    def get_total_fees(self, obj):
        # Calculate total fees from all invoices (simple sum for now)
        return obj.invoices.aggregate(Sum('total_amount'))['total_amount__sum'] or 0

    def get_total_paid(self, obj):
        # Calculate total paid from all payments
        return obj.invoices.aggregate(Sum('total_paid'))['total_paid__sum'] or 0

    def get_total_remaining(self, obj):
        # Calculate total remaining by subtracting total paid from total fees
        total_fees = obj.invoices.aggregate(Sum('total_amount'))['total_amount__sum'] or 0
        total_paid = obj.invoices.aggregate(Sum('total_paid'))['total_paid__sum'] or 0
        return total_fees - total_paid

class ApplicationSerializer(serializers.ModelSerializer):
    class Meta:
        model = Application
        # Update fields to include the four name parts
        fields = '__all__' # Or list them explicitly: ('id', 'first_name_ar', 'second_name_ar', 'third_name_ar', 'fourth_name_ar', 'first_name_en', 'second_name_en', 'third_name_en', 'fourth_name_en', ...)

class AdmissionDecisionSerializer(serializers.ModelSerializer):
    application_full_name_ar = serializers.SerializerMethodField()
    admitted_student_full_name_ar = serializers.SerializerMethodField()

    class Meta:
        model = AdmissionDecision
        fields = '__all__'

    def get_application_full_name_ar(self, obj):
        return f"{obj.application.first_name_ar} {obj.application.second_name_ar} {obj.application.third_name_ar} {obj.application.fourth_name_ar}"

    def get_admitted_student_full_name_ar(self, obj):
        if obj.admitted_student:
            return f"{obj.admitted_student.first_name_ar} {obj.admitted_student.second_name_ar} {obj.admitted_student.third_name_ar} {obj.admitted_student.fourth_name_ar}"
        return None

class SemesterEnrollmentSerializer(serializers.ModelSerializer):
    student_full_name_ar = serializers.SerializerMethodField()
    academic_year_name = serializers.CharField(source='academic_year.year', read_only=True)
    academic_semester_name_ar = serializers.CharField(source='academic_semester.name_ar', read_only=True)

    class Meta:
        model = SemesterEnrollment
        fields = '__all__'

    def get_student_full_name_ar(self, obj):
        return f"{obj.student.first_name_ar} {obj.student.second_name_ar} {obj.student.third_name_ar} {obj.student.fourth_name_ar}"
class StudentRegistrationSerializer(serializers.Serializer):
    student_id = serializers.CharField(max_length=20, required=True, help_text=_("الرقم الجامعي للطالب"))
    academic_year_id = serializers.IntegerField(required=True, help_text=_("معرف السنة الأكاديمية التي سيسجل فيها الطالب"))
    academic_semester_id = serializers.IntegerField(required=True, help_text=_("معرف الفصل الدراسي الذي سيسجل فيه الطالب"))
    payment_verified = serializers.BooleanField(default=False, help_text=_("هل تم التحقق من الدفع؟"))

class AcceptedStudentsImportSerializer(serializers.Serializer):
    file = serializers.FileField(help_text=_("ملف Excel/CSV لقائمة الطلاب المقبولين"))
class RegistrarApprovalSerializer(serializers.Serializer):
    student_id = serializers.CharField(required=True, help_text=_("الرقم الجامعي للطالب المراد اعتماده"))
    documents_approved_status = serializers.BooleanField(required=True, help_text=_("حالة اعتماد المستندات (صحيح/خطأ)"))
    # Optionally, allow registrar to set official university student_id if not already set,
    # or assign initial academic level if needed at this stage.

