# student_management/models.py
from django.utils.translation import gettext_lazy as _ 
from django.db import models
from users.models import User  # Correctly import your custom User model
from university_structure.models import Department, Batch, AcademicYear, AcademicSemester # Import necessary models
# Add this import

class Student(models.Model):
    user = models.OneToOneField(User, on_delete=models.CASCADE, null=True, blank=True,
                                verbose_name=_("حساب المستخدم"), related_name='student_profile')
    student_id = models.CharField(max_length=20, unique=True, null=True, blank=True, verbose_name=_("الرقم الجامعي")) 
    
    first_name_ar = models.CharField(max_length=100, verbose_name=_("الاسم الأول (عربي)"))
    second_name_ar = models.CharField(max_length=100, verbose_name=_("الاسم الثاني (عربي)"))
    third_name_ar = models.CharField(max_length=100, verbose_name=_("الاسم الثالث (عربي)"))
    fourth_name_ar = models.CharField(max_length=100, verbose_name=_("الاسم الرابع (عربي)"))
    
    first_name_en = models.CharField(max_length=100, verbose_name=_("First Name (English)"))
    second_name_en = models.CharField(max_length=100, verbose_name=_("Second Name (English)"))
    third_name_en = models.CharField(max_length=100, verbose_name=_("Third Name (English)"))
    fourth_name_en = models.CharField(max_length=100, verbose_name=_("Fourth Name (English)"))
    national_id = models.CharField(max_length=20, unique=True, null=True, blank=True, verbose_name=_("الرقم الوطني"))

    date_of_birth = models.DateField(null=True, blank=True, verbose_name=_("تاريخ الميلاد"))
    
    GENDER_CHOICES = [
        ('Male', _('ذكر')),
        ('Female', _('أنثى')),
        ('Other', _('أخرى')),
    ]
    gender = models.CharField(max_length=10, choices=GENDER_CHOICES, null=True, blank=True, verbose_name=_("الجنس"))
    
    nationality = models.CharField(max_length=100, null=True, blank=True, verbose_name=_("الجنسية"))
    
    contact_email = models.EmailField(unique=True, verbose_name=_("البريد الإلكتروني للاتصال"))
    contact_phone = models.CharField(max_length=20, blank=True, null=True, verbose_name=_("رقم الهاتف"))
    address = models.TextField(blank=True, null=True, verbose_name=_("العنوان"))
    
    department = models.ForeignKey(Department, on_delete=models.SET_NULL, null=True, blank=True,
                                   related_name='students', verbose_name=_("القسم"))
    batch = models.ForeignKey(Batch, on_delete=models.SET_NULL, null=True, blank=True,
                              related_name='students', verbose_name=_("الدفعة"))
    
    admission_date = models.DateField(null=True, blank=True, verbose_name=_("تاريخ القبول الأولي"))
    admission_source = models.CharField(max_length=50, default='Ministry', verbose_name=_("مصدر القبول"))

    STATUS_CHOICES = [
        ('Provisional', _('مقبول مبدئياً')),
        ('Active', _('طالب حالي')),
        ('Inactive', _('غير نشط')),
        ('Graduated', _('متخرج')),
        ('Suspended', _('موقوف')),
        ('Expelled', _('مفصول')),
        ('On Leave', _('في إجازة')),
        ('Withdrawn', _('منسحب')),
        ('Rejected', _('مرفوض')),
    ]
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='Provisional', verbose_name=_("حالة الطالب"))

    id_card_uploaded = models.BooleanField(default=False, verbose_name=_("تم رفع الهوية"))
    high_school_cert_uploaded = models.BooleanField(default=False, verbose_name=_("تم رفع شهادة الثانوية"))
    personal_photo_uploaded = models.BooleanField(default=False, verbose_name=_("تم رفع الصورة الشخصية"))
    documents_approved = models.BooleanField(default=False, verbose_name=_("تم اعتماد المستندات"))

    current_academic_year = models.ForeignKey(AcademicYear, on_delete=models.SET_NULL, null=True, blank=True,
                                              related_name='students_in_year', verbose_name=_("السنة الأكاديمية الحالية"))
    current_academic_semester = models.ForeignKey(AcademicSemester, on_delete=models.SET_NULL, null=True, blank=True,
                                                  related_name='students_in_semester', verbose_name=_("الفصل الدراسي الحالي"))
    academic_level = models.PositiveSmallIntegerField(default=0, verbose_name=_("المستوى الأكاديمي الحالي"))
    
    is_registered_for_current_semester = models.BooleanField(default=False, verbose_name=_("مسجل للفصل الحالي"))
    
    class Meta:
        verbose_name = _("طالب")
        verbose_name_plural = _("الطلاب")

    # Corrected: Remove the recursive get_status_display property
    # Django provides get_FOO_display() automatically for fields with choices.
    @property
    def full_name_ar(self):
        return f"{self.first_name_ar} {self.second_name_ar} {self.third_name_ar} {self.fourth_name_ar}"

    @property
    def full_name_en(self):
        return f"{self.first_name_en} {self.second_name_en} {self.third_name_en} {self.fourth_name_en}"

    def __str__(self):
        return f"{self.student_id or _('مبدئي')} - {self.full_name_ar} ({self.status})"
# New model for public application form submissions
class Application(models.Model):
    GENDER_CHOICES = [
        ('Male', _('ذكر')),
        ('Female', _('أنثى')),
        ('Other', _('أخرى')),
    ]
    STATUS_CHOICES = [
        ('Pending', _('قيد المراجعة')),
        ('Accepted', _('مقبول')),
        ('Rejected', _('مرفوض')),
    ]
    
    # Personal Information
    first_name_ar = models.CharField(max_length=50, verbose_name=_("الاسم الأول (عربي)"))
    second_name_ar = models.CharField(max_length=50, blank=True, null=True, verbose_name=_("الاسم الثاني (عربي)"))
    third_name_ar = models.CharField(max_length=50, blank=True, null=True, verbose_name=_("الاسم الثالث (عربي)"))
    fourth_name_ar = models.CharField(max_length=50, verbose_name=_("الاسم الرابع (عربي)"))
    first_name_en = models.CharField(max_length=50, verbose_name=_("First Name (English)"))
    second_name_en = models.CharField(max_length=50, blank=True, null=True, verbose_name=_("Second Name (English)"))
    third_name_en = models.CharField(max_length=50, blank=True, null=True, verbose_name=_("Third Name (English)"))
    fourth_name_en = models.CharField(max_length=50, verbose_name=_("Fourth Name (English)"))

    email = models.EmailField(unique=True, verbose_name=_("البريد الإلكتروني"))
    phone_number = models.CharField(max_length=20, blank=True, null=True, verbose_name=_("رقم الهاتف"))
    date_of_birth = models.DateField(verbose_name=_("تاريخ الميلاد"))
    gender = models.CharField(max_length=10, choices=GENDER_CHOICES, verbose_name=_("النوع"))
    nationality = models.CharField(max_length=50, verbose_name=_("الجنسية"))
    address = models.TextField(blank=True, null=True, verbose_name="العنوان")

    # Academic Information
    high_school_certificate_score = models.DecimalField(max_digits=5, decimal_places=2, verbose_name=_("درجة الشهادة الثانوية"))
    previous_school = models.CharField(max_length=255, blank=True, null=True, verbose_name=_("المدرسة السابقة"))
    desired_department = models.ForeignKey(Department, on_delete=models.SET_NULL, null=True, verbose_name=_("القسم المرغوب"))

    # New file fields for documents
    id_card_file = models.FileField(upload_to='application_docs/id_cards/', blank=True, null=True, verbose_name=_("صورة الهوية"))
    high_school_cert_file = models.FileField(upload_to='application_docs/certs/', blank=True, null=True, verbose_name=_("صورة الشهادة الثانوية"))

    # Payment Information
    application_fee_paid = models.BooleanField(default=False, verbose_name=_("تم دفع رسوم التقديم"))
    payment_reference_number = models.CharField(max_length=100, blank=True, null=True, verbose_name=_("رقم العملية البنكية"))
    payment_receipt_file = models.FileField(upload_to='application_docs/payments/', blank=True, null=True, verbose_name=_("إيصال الدفع"))
    
    # Application Status
    application_date = models.DateTimeField(auto_now_add=True, verbose_name=_("تاريخ التقديم"))
    application_status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='Pending', verbose_name=_("حالة الطلب"))

    class Meta:
        verbose_name = _("طلب تقديم")
        verbose_name_plural = _("طلبات التقديم")
        ordering = ['-application_date']
    
    def __str__(self):
        return f"طلب {self.first_name_ar} - {self.application_status}"

class AdmissionDecision(models.Model):
    """
    Records the admission decision for an application.
    """
    application = models.OneToOneField(Application, on_delete=models.CASCADE, related_name='admission_decision',
                                       verbose_name=_("طلب التقديم"))
    
    DECISION_CHOICES = [
        ('Accepted', _('مقبول')),
        ('Rejected', _('مرفوض')),
        ('Waitlisted', _('قائمة الانتظار')),
    ]
    decision = models.CharField(max_length=20, choices=DECISION_CHOICES, verbose_name=_("القرار"))
    
    admitted_to_department = models.ForeignKey(Department, on_delete=models.SET_NULL, null=True, blank=True,
                                               verbose_name=_("القبول بقسم"))
    admitted_to_batch = models.ForeignKey(Batch, on_delete=models.SET_NULL, null=True, blank=True,
                                             verbose_name=_("القبول بدفعة"))
    decision_date = models.DateField(auto_now_add=True, verbose_name="تاريخ القرار")
    notes = models.TextField(blank=True, null=True, verbose_name="ملاحظات")

    admitted_student = models.OneToOneField(Student, on_delete=models.SET_NULL, null=True, blank=True,
                                            verbose_name=_("الطالب المقبول"))

    class Meta:
        verbose_name = "قرار قبول"
        verbose_name_plural = "قرارات القبول"

    def __str__(self):
        return f"قرار {self.application.first_name_ar}: {self.decision}"
class SemesterEnrollment(models.Model):
    """
    Records a student's enrollment in a specific semester.
    Automatically enrolls students in fixed courses for their current level and semester.
    """
    student = models.ForeignKey(Student, on_delete=models.CASCADE, related_name='enrollments', verbose_name="الطالب")
    academic_year = models.ForeignKey(AcademicYear, on_delete=models.CASCADE, verbose_name="السنة الأكاديمية")
    academic_semester = models.ForeignKey(AcademicSemester, on_delete=models.CASCADE, verbose_name="الفصل الدراسي")
    enrollment_date = models.DateField(auto_now_add=True, verbose_name="تاريخ التسجيل")
    is_active = models.BooleanField(default=True, verbose_name="نشط في هذا الفصل")
    
    # Financial status linked to registration enforcement
    fees_paid = models.BooleanField(default=False, verbose_name="تم دفع الرسوم")

    class Meta:
        unique_together = ('student', 'academic_semester') # A student can only enroll once per semester
        verbose_name = "تسجيل فصلي"
        verbose_name_plural = "تسجيلات فصلية"

    def __str__(self):
        # Displays the full Arabic name from the student, then the semester
        return f"{self.student.first_name_ar} {self.student.second_name_ar} {self.student.third_name_ar} {self.student.fourth_name_ar} - {self.academic_semester}"