# finance_management/views.py
from django.db.models import Q
from rest_framework import viewsets, status
from rest_framework.response import Response
from rest_framework.views import APIView
from rest_framework.parsers import MultiPartParser, FormParser
from rest_framework.permissions import IsAuthenticated, IsAdminUser # Added IsAdminUser
from django.shortcuts import get_object_or_404
from django.utils.translation import gettext_lazy as _
from django.db import transaction, models
from student_management.serializers import StudentSerializer
import traceback
from django_filters.rest_framework import DjangoFilterBackend # <--- ADD THIS IMPORT
from rest_framework import filters
import random
from .models import FeeType, FeeStructure, StudentInvoice, InvoiceItem, StudentPayment
from .serializers import FeeTypeSerializer, FeeStructureSerializer, StudentInvoiceSerializer, InvoiceItemSerializer, StudentPaymentSerializer, StaffPaymentRecordSerializer
from student_management.models import Student
from university_structure.models import AcademicYear, AcademicSemester, Batch
from users.models import FacultyProfile


# ViewSets for CRUD operations on Finance Management models
class FeeTypeViewSet(viewsets.ModelViewSet):
    queryset = FeeType.objects.all().order_by('name_ar')
    serializer_class = FeeTypeSerializer
    permission_classes = [IsAuthenticated, IsAdminUser] # Restrict to staff/admin

class FeeStructureViewSet(viewsets.ModelViewSet):
    queryset = FeeStructure.objects.all().order_by('academic_year__year', 'academic_semester__semester_number', 'fee_type__name_ar')
    serializer_class = FeeStructureSerializer
    permission_classes = [IsAuthenticated, IsAdminUser]

class StudentInvoiceViewSet(viewsets.ModelViewSet):
    queryset = StudentInvoice.objects.all().order_by('-generated_at')
    serializer_class = StudentInvoiceSerializer
    permission_classes = [IsAuthenticated, IsAdminUser]
    filter_backends = [DjangoFilterBackend, filters.SearchFilter]
    filterset_fields = ['student', 'academic_year', 'academic_semester', 'status']
    search_fields = ['invoice_number', 'student__student_id', 'student__first_name_ar', 'student__fourth_name_ar']

    def get_serializer_context(self):
        return {'request': self.request}


class InvoiceItemViewSet(viewsets.ModelViewSet):
    queryset = InvoiceItem.objects.all()
    serializer_class = InvoiceItemSerializer
    permission_classes = [IsAuthenticated, IsAdminUser]

class StudentPaymentViewSet(viewsets.ModelViewSet):
    queryset = StudentPayment.objects.all().order_by('-payment_date')
    serializer_class = StudentPaymentSerializer
    permission_classes = [IsAuthenticated, IsAdminUser] # Restrict to staff/admin
    filter_backends = [DjangoFilterBackend, filters.SearchFilter]
    filterset_fields = ['invoice__student', 'invoice__academic_year', 'invoice__academic_semester', 'status']
    search_fields = ['invoice__invoice_number', 'invoice__student__student_id', 'invoice__student__first_name_ar', 'invoice__student__fourth_name_ar', 'reference_number']

    def get_serializer_context(self): # Pass request context to serializer for file URL
        return {'request': self.request}


# API for Student to Upload Payment Receipt
class PaymentUploadAPIView(APIView):
    parser_classes = (MultiPartParser, FormParser)
    permission_classes = [IsAuthenticated] # Student can upload their own receipt

    def post(self, request, *args, **kwargs):
        invoice_id = request.data.get('invoice_id')
        payment_amount = request.data.get('amount')
        payment_method = request.data.get('payment_method', 'Bank Transfer')
        reference_number = request.data.get('reference_number', None)
        payment_receipt_file = request.FILES.get('payment_receipt_file')

        if not invoice_id or not payment_amount or not payment_receipt_file:
            return Response({'detail': _("الرجاء تقديم معرف الفاتورة، المبلغ، وإيصال الدفع.")}, status=status.HTTP_400_BAD_REQUEST)

        try:
            invoice = get_object_or_404(StudentInvoice, id=invoice_id)
            
            if invoice.student.user != request.user: # Ensure student owns the invoice
                return Response({'detail': _("غير مصرح لك برفع إيصال دفع لهذه الفاتورة.")}, status=status.HTTP_403_FORBIDDEN)

            with transaction.atomic():
                payment = StudentPayment.objects.create(
                    invoice=invoice,
                    amount=payment_amount,
                    payment_method=payment_method,
                    reference_number=reference_number,
                    payment_receipt_file=payment_receipt_file,
                    status='Pending'
                )

            return Response({'detail': _("تم رفع إيصال الدفع بنجاح. سيتم مراجعته من قبل مسؤول القبول."), 'payment_id': payment.id}, status=status.HTTP_200_OK)

        except StudentInvoice.DoesNotExist:
            return Response({'detail': _("الفاتورة غير موجودة.")}, status=status.HTTP_404_NOT_FOUND)
        except Exception as e:
            traceback.print_exc()
            return Response({'detail': str(e)}, status=status.HTTP_500_INTERNAL_SERVER_ERROR)


# API for Registrar to Approve/Reject Payments
class ApprovePaymentAPIView(APIView):
    permission_classes = [IsAuthenticated, IsAdminUser] # Only authenticated staff/admin can approve

    def post(self, request, *args, **kwargs):
        payment_id = request.data.get('payment_id')
        action_status = request.data.get('status') # 'Approved' or 'Rejected'

        if not payment_id or action_status not in ['Approved', 'Rejected']:
            return Response({'detail': _("الرجاء تقديم معرف الدفعة والحالة المطلوبة (Approved أو Rejected).")}, status=status.HTTP_400_BAD_REQUEST)

        try:
            payment = get_object_or_404(StudentPayment, id=payment_id)
            
            # Ensure the user has registrar permission (optional, IsAdminUser covers staff)
            if not hasattr(request.user, 'faculty_profile') or not request.user.faculty_profile.is_registrar:
                return Response({'detail': _("ليس لديك صلاحيات المسجل لاعتماد الدفعات.")}, status=status.HTTP_403_FORBIDDEN)

            with transaction.atomic():
                payment.status = action_status
                payment.save()

                # Optional: Update related invoice status if payment is Approved and covers full amount
                if action_status == 'Approved':
                    invoice = payment.invoice
                    if invoice.get_remaining_amount() <= 0: # If full amount paid/covered
                        invoice.status = 'Paid'
                    elif invoice.get_paid_amount() > 0:
                        invoice.status = 'Partially Paid'
                    invoice.save()

            return Response({'detail': _("تم تحديث حالة الدفعة بنجاح إلى ") + _(action_status) + "."}, status=status.HTTP_200_OK)

        except StudentPayment.DoesNotExist:
            return Response({'detail': _("الدفعة غير موجودة.")}, status=status.HTTP_404_NOT_FOUND)
        except Exception as e:
            traceback.print_exc()
            return Response({'detail': str(e)}, status=status.HTTP_500_INTERNAL_SERVER_ERROR)
class StaffRecordPaymentAPIView(APIView):
    parser_classes = (MultiPartParser, FormParser)
    permission_classes = [IsAuthenticated, IsAdminUser]

    def post(self, request, *args, **kwargs):
        if not hasattr(request.user, 'faculty_profile') or not request.user.faculty_profile.is_finance_officer:
            return Response({'detail': _("ليس لديك صلاحيات مسؤول المالية لتسجيل الدفعات.")}, status=status.HTTP_403_FORBIDDEN)

        serializer = StaffPaymentRecordSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        student_id_from_form = serializer.validated_data['student_id'] # This is now the database ID
        invoice_id = serializer.validated_data['invoice_id']
        amount = serializer.validated_data['amount']
        payment_method = serializer.validated_data['payment_method']
        reference_number = serializer.validated_data['reference_number']
        payment_receipt_file = serializer.validated_data.get('payment_receipt_file')

        try:
            # CRITICAL FIX: Find the student by their database ID
            student = get_object_or_404(Student, id=student_id_from_form)
            invoice = get_object_or_404(StudentInvoice, id=invoice_id, student=student) # Ensure invoice belongs to student

            with transaction.atomic():
                payment = StudentPayment.objects.create(
                    invoice=invoice,
                    amount=amount,
                    payment_method=payment_method,
                    reference_number=reference_number,
                    payment_receipt_file=payment_receipt_file,
                    status='Approved',
                    received_by=request.user.faculty_profile
                )

            return Response({'detail': _("تم تسجيل الدفعة بنجاح. وتم اعتمادها تلقائياً."), 'payment_id': payment.id}, status=status.HTTP_200_OK)

        except Student.DoesNotExist:
            return Response({'detail': _("الطالب غير موجود.")}, status=status.HTTP_404_NOT_FOUND)
        except StudentInvoice.DoesNotExist:
            return Response({'detail': _("الفاتورة غير موجودة أو لا تخص هذا الطالب.")}, status=status.HTTP_404_NOT_FOUND)
        except Exception as e:
            traceback.print_exc()
            return Response({'detail': str(e)}, status=status.HTTP_500_INTERNAL_SERVER_ERROR)

class GenerateInvoicesAPIView(APIView):
    permission_classes = [IsAuthenticated, IsAdminUser]

    def post(self, request, *args, **kwargs):
        if not hasattr(request.user, 'faculty_profile') or not request.user.faculty_profile.is_finance_officer:
            return Response({'detail': _("ليس لديك صلاحيات مسؤول المالية لإنشاء الفواتير.")}, status=status.HTTP_403_FORBIDDEN)

        batch_id = request.data.get('batch_id')
        academic_semester_id = request.data.get('academic_semester_id')

        if not batch_id or not academic_semester_id:
            return Response({'detail': _("الرجاء تقديم معرف الدفعة والفصل الدراسي الأكاديمي.")}, status=status.HTTP_400_BAD_REQUEST)

        try:
            batch = get_object_or_404(Batch, id=batch_id)
            academic_semester = get_object_or_404(AcademicSemester, id=academic_semester_id)
            students = Student.objects.filter(batch=batch)

            if not students.exists():
                return Response({'detail': _("لا يوجد طلاب في هذه الدفعة.")}, status=status.HTTP_404_NOT_FOUND)

            invoices_created = 0
            with transaction.atomic():
                # Find fee structures for this batch/semester
                fee_structures = FeeStructure.objects.filter(
                    models.Q(academic_year=academic_semester.academic_year) | models.Q(academic_year=None),
                    models.Q(academic_semester=academic_semester) | models.Q(academic_semester=None),
                    models.Q(department=batch.department) | models.Q(department=None),
                    models.Q(batch=batch) | models.Q(batch=None)
                )

                if not fee_structures.exists():
                    return Response({'detail': _("لم يتم العثور على هياكل رسوم مطابقة لهذه الدفعة والفصل الدراسي.")}, status=status.HTTP_404_NOT_FOUND)

                for student in students:
                    # Check if an invoice for this student/semester already exists to prevent duplicates
                    if not StudentInvoice.objects.filter(student=student, academic_semester=academic_semester).exists():
                        
                        total_amount = sum(s.amount for s in fee_structures)
                        invoice_number = f"INV-{student.student_id}-{academic_semester.academic_year.year}-{academic_semester.semester_number}"
                        
                        invoice = StudentInvoice.objects.create(
                            student=student,
                            academic_year=academic_semester.academic_year,
                            academic_semester=academic_semester,
                            invoice_number=invoice_number,
                            total_amount=total_amount,
                            due_date=academic_semester.registration_end_date,
                            status='Pending'
                        )
                        
                        for structure in fee_structures:
                            InvoiceItem.objects.create(
                                invoice=invoice,
                                fee_type=structure.fee_type,
                                amount=structure.amount,
                                description_ar=structure.fee_type.description_ar
                            )
                        invoices_created += 1

            return Response({'detail': _("تم إنشاء عدد %d فاتورة بنجاح." % invoices_created)}, status=status.HTTP_200_OK)

        except Batch.DoesNotExist:
            return Response({'detail': _("الدفعة غير موجودة.")}, status=status.HTTP_404_NOT_FOUND)
        except AcademicSemester.DoesNotExist:
            return Response({'detail': _("الفصل الدراسي غير موجود.")}, status=status.HTTP_404_NOT_FOUND)
        except Exception as e:
            traceback.print_exc()
            return Response({'detail': str(e)}, status=status.HTTP_500_INTERNAL_SERVER_ERROR)

class StudentSearchAPIView(APIView):
    permission_classes = [IsAuthenticated, IsAdminUser]
    
    def get(self, request, *args, **kwargs):
        query = request.query_params.get('query', '')
        department_id = request.query_params.get('department_id', None)
        academic_level = request.query_params.get('academic_level', None)
        
        filters = Q()
        if department_id:
            filters &= Q(department__id=department_id)
        if academic_level:
            filters &= Q(academic_level=academic_level)
            
        students = Student.objects.filter(filters).order_by('full_name_ar') # Order by full name
        
        serializer = StudentSerializer(students, many=True)
        return Response(serializer.data)
    
class ManualInvoiceCreateAPIView(APIView): # <--- New API for manual invoice creation
    permission_classes = [IsAuthenticated, IsAdminUser]

    def post(self, request, *args, **kwargs):
        if not hasattr(request.user, 'faculty_profile') or not request.user.faculty_profile.is_finance_officer:
            return Response({'detail': _("ليس لديك صلاحيات مسؤول المالية لإنشاء الفواتير.")}, status=status.HTTP_403_FORBIDDEN)
        
        student_id = request.data.get('student_id')
        academic_year_id = request.data.get('academic_year_id')
        academic_semester_id = request.data.get('academic_semester_id')
        total_amount = request.data.get('total_amount')

        if not student_id or not academic_year_id or not academic_semester_id or not total_amount:
            return Response({'detail': _("الرجاء تقديم معرف الطالب، السنة الأكاديمية، الفصل الدراسي والمبلغ الإجمالي.")}, status=status.HTTP_400_BAD_REQUEST)

        try:
            student = get_object_or_404(Student, id=student_id) # Find student by ID
            academic_year = get_object_or_404(AcademicYear, id=academic_year_id)
            academic_semester = get_object_or_404(AcademicSemester, id=academic_semester_id)

            invoice_number = f"INV-{student.student_id}-{academic_year.year}-{academic_semester.semester_number}-{random.randint(1000, 9999)}"
            
            with transaction.atomic():
                invoice = StudentInvoice.objects.create(
                    student=student,
                    academic_year=academic_year,
                    academic_semester=academic_semester,
                    invoice_number=invoice_number,
                    total_amount=total_amount,
                    due_date=academic_semester.registration_end_date,
                    status='Pending'
                )

                # Optional: You might want to create a default invoice item here
                # Example:
                # fee_type, _ = FeeType.objects.get_or_create(name_en='Manual Fee', defaults={'name_ar': 'رسوم يدوية'})
                # InvoiceItem.objects.create(
                #     invoice=invoice,
                #     fee_type=fee_type,
                #     amount=total_amount,
                #     description_ar='رسوم تم إنشاؤها يدوياً'
                # )

            return Response({'detail': _("تم إنشاء الفاتورة اليدوية بنجاح."), 'invoice_id': invoice.id}, status=status.HTTP_200_OK)

        except Student.DoesNotExist:
            return Response({'detail': _("الطالب غير موجود.")}, status=status.HTTP_404_NOT_FOUND)
        except AcademicYear.DoesNotExist:
            return Response({'detail': _("السنة الأكاديمية غير موجودة.")}, status=status.HTTP_404_NOT_FOUND)
        except AcademicSemester.DoesNotExist:
            return Response({'detail': _("الفصل الدراسي غير موجود.")}, status=status.HTTP_404_NOT_FOUND)
        except Exception as e:
            traceback.print_exc()
            return Response({'detail': str(e)}, status=status.HTTP_500_INTERNAL_SERVER_ERROR)
