# finance_management/serializers.py
from django.utils.translation import gettext_lazy as _ 
from rest_framework import serializers
from .models import FeeType, FeeStructure, StudentInvoice, InvoiceItem, StudentPayment
from student_management.models import Student

from users.models import FacultyProfile

class FeeTypeSerializer(serializers.ModelSerializer):
    class Meta:
        model = FeeType
        fields = '__all__'

class FeeStructureSerializer(serializers.ModelSerializer):
    fee_type_name_ar = serializers.CharField(source='fee_type.name_ar', read_only=True)
    academic_year_name = serializers.CharField(source='academic_year.year', read_only=True, allow_null=True)
    academic_semester_name_ar = serializers.CharField(source='academic_semester.name_ar', read_only=True, allow_null=True)
    department_name_ar = serializers.CharField(source='department.name_ar', read_only=True, allow_null=True)
    batch_name = serializers.CharField(source='batch.name', read_only=True, allow_null=True)

    class Meta:
        model = FeeStructure
        fields = '__all__'

class InvoiceItemSerializer(serializers.ModelSerializer):
    fee_type_name_ar = serializers.CharField(source='fee_type.name_ar', read_only=True, allow_null=True)

    class Meta:
        model = InvoiceItem
        fields = '__all__'

class StudentPaymentSerializer(serializers.ModelSerializer):
    invoice_number = serializers.CharField(source='invoice.invoice_number', read_only=True)
    student_full_name_ar = serializers.SerializerMethodField()
    received_by_name_ar = serializers.CharField(source='received_by.full_name_ar', read_only=True, allow_null=True)
    status_display = serializers.CharField(source='get_status_display', read_only=True)
    
    payment_receipt_file_url = serializers.SerializerMethodField() # For file URL

    class Meta:
        model = StudentPayment
        fields = '__all__'

    def get_student_full_name_ar(self, obj):
        return f"{obj.invoice.student.first_name_ar} {obj.invoice.student.fourth_name_ar}"
    
    def get_payment_receipt_file_url(self, obj):
        if obj.payment_receipt_file and self.context.get('request'): # Check if file exists and request context is available
            return self.context['request'].build_absolute_uri(obj.payment_receipt_file.url)
        return None

class StudentInvoiceSerializer(serializers.ModelSerializer):
    student_full_name_ar = serializers.SerializerMethodField()
    academic_year_name = serializers.CharField(source='academic_year.year', read_only=True)
    academic_semester_name_ar = serializers.CharField(source='academic_semester.name_ar', read_only=True)
    status_display = serializers.CharField(source='get_status_display', read_only=True)
    
    items = InvoiceItemSerializer(many=True, read_only=True)
    payments = StudentPaymentSerializer(many=True, read_only=True, context={'request': None})
    
    # CRITICAL FIX: Rename these fields to avoid conflict
    paid_amount = serializers.DecimalField(source='get_paid_amount', max_digits=10, decimal_places=2, read_only=True)
    remaining_amount = serializers.DecimalField(source='get_remaining_amount', max_digits=10, decimal_places=2, read_only=True)

    class Meta:
        model = StudentInvoice
        # CRITICAL FIX: Make sure the new field names are in the 'fields' tuple
        fields = '__all__'

    def get_student_full_name_ar(self, obj):
        return f"{obj.student.first_name_ar} {obj.student.fourth_name_ar}"
class StaffPaymentRecordSerializer(serializers.Serializer):
    student_id = serializers.IntegerField(required=True, help_text=_("معرف الطالب")) # <--- Changed to IntegerField for DB ID
    invoice_id = serializers.IntegerField(required=True, help_text=_("معرف الفاتورة"))
    amount = serializers.DecimalField(max_digits=10, decimal_places=2, required=True, help_text=_("المبلغ المدفوع"))
    payment_method = serializers.CharField(required=True, help_text=_("طريقة الدفع"))
    reference_number = serializers.CharField(required=False, allow_blank=True, help_text=_("الرقم المرجعي"))
    payment_receipt_file = serializers.FileField(required=False, allow_null=True, help_text=_("إيصال الدفع (اختياري)"))