# finance_management/models.py
from django.db import models
from django.utils.translation import gettext_lazy as _
from student_management.models import Student
from university_structure.models import AcademicYear, AcademicSemester, Department, Batch
from users.models import FacultyProfile

class FeeType(models.Model):
    name_ar = models.CharField(max_length=100, unique=True, verbose_name=_("اسم نوع الرسوم (عربي)"))
    name_en = models.CharField(max_length=100, unique=True, verbose_name=_("اسم نوع الرسوم (إنجليزي)"))
    description_ar = models.TextField(blank=True, null=True, verbose_name=_("الوصف (عربي)"))
    is_active = models.BooleanField(default=True, verbose_name=_("نشط"))
    
    class Meta:
        verbose_name = _("نوع رسوم")
        verbose_name_plural = _("أنواع الرسوم")

    def __str__(self):
        return self.name_ar

class FeeStructure(models.Model):
    fee_type = models.ForeignKey(FeeType, on_delete=models.CASCADE, related_name='structures', verbose_name=_("نوع الرسوم"))
    amount = models.DecimalField(max_digits=10, decimal_places=2, verbose_name=_("المبلغ"))
    academic_year = models.ForeignKey(AcademicYear, on_delete=models.SET_NULL, null=True, blank=True, verbose_name=_("السنة الأكاديمية"))
    academic_semester = models.ForeignKey(AcademicSemester, on_delete=models.SET_NULL, null=True, blank=True, verbose_name=_("الفصل الدراسي"))
    department = models.ForeignKey(Department, on_delete=models.SET_NULL, null=True, blank=True, verbose_name=_("القسم"))
    batch = models.ForeignKey(Batch, on_delete=models.SET_NULL, null=True, blank=True, verbose_name=_("الدفعة"))

    class Meta:
        verbose_name = _("هيكل رسوم")
        verbose_name_plural = _("هياكل الرسوم")
        unique_together = ('fee_type', 'academic_year', 'academic_semester', 'department', 'batch')

    def __str__(self):
        return f"{self.fee_type.name_ar} - {self.amount}"

class StudentInvoice(models.Model):
    INVOICE_STATUS_CHOICES = [
        ('Pending', _('معلقة')),
        ('Paid', _('مدفوعة')),
        ('Partially Paid', _('مدفوعة جزئياً')),
        ('Overdue', _('متأخرة')),
        ('Cancelled', _('ملغاة')),
    ]
    student = models.ForeignKey(Student, on_delete=models.CASCADE, related_name='invoices', verbose_name=_("الطالب"))
    academic_year = models.ForeignKey(AcademicYear, on_delete=models.CASCADE, verbose_name=_("السنة الأكاديمية"))
    academic_semester = models.ForeignKey(AcademicSemester, on_delete=models.CASCADE, verbose_name=_("الفصل الدراسي"))
    invoice_number = models.CharField(max_length=50, unique=True, verbose_name=_("رقم الفاتورة"))
    total_amount = models.DecimalField(max_digits=10, decimal_places=2, verbose_name=_("المبلغ الإجمالي"))
    total_paid = models.DecimalField(max_digits=10, decimal_places=2, default=0, verbose_name=_("إجمالي المدفوع"))
    due_date = models.DateField(verbose_name=_("تاريخ الاستحقاق"))
    status = models.CharField(max_length=20, choices=INVOICE_STATUS_CHOICES, default='Pending', verbose_name=_("حالة الفاتورة"))
    generated_at = models.DateTimeField(auto_now_add=True, verbose_name=_("تاريخ الإنشاء"))

    def get_paid_amount(self):
        return self.payments.aggregate(models.Sum('amount'))['amount__sum'] or 0
    
    def get_remaining_amount(self):
        return self.total_amount - self.get_paid_amount()

    class Meta:
        verbose_name = _("فاتورة طالب")
        verbose_name_plural = _("فواتير الطلاب")
        ordering = ['-generated_at']

    def __str__(self):
        return self.invoice_number

class InvoiceItem(models.Model):
    invoice = models.ForeignKey(StudentInvoice, on_delete=models.CASCADE, related_name='items', verbose_name=_("الفاتورة"))
    fee_type = models.ForeignKey(FeeType, on_delete=models.SET_NULL, null=True, verbose_name=_("نوع الرسوم"))
    amount = models.DecimalField(max_digits=10, decimal_places=2, verbose_name=_("المبلغ"))
    description_ar = models.TextField(blank=True, null=True, verbose_name=_("الوصف (عربي)"))

    class Meta:
        verbose_name = _("بند فاتورة")
        verbose_name_plural = _("بنود الفواتير")

    def __str__(self):
        return f"{self.invoice.invoice_number} - {self.fee_type.name_ar}"

class StudentPayment(models.Model):
    PAYMENT_METHOD_CHOICES = [
        ('Cash', _('نقداً')),
        ('Bank Transfer', _('تحويل بنكي')),
        ('Online', _('دفع إلكتروني')),
        ('Other', _('أخرى')),
    ]
    STATUS_CHOICES = [
        ('Pending', _('قيد المراجعة')),
        ('Approved', _('معتمدة')),
        ('Rejected', _('مرفوضة')),
    ]
    invoice = models.ForeignKey(StudentInvoice, on_delete=models.CASCADE, related_name='payments', verbose_name=_("الفاتورة"))
    amount = models.DecimalField(max_digits=10, decimal_places=2, verbose_name=_("المبلغ المدفوع"))
    payment_date = models.DateTimeField(auto_now_add=True, verbose_name=_("تاريخ الدفع"))
    payment_method = models.CharField(max_length=20, choices=PAYMENT_METHOD_CHOICES, verbose_name=_("طريقة الدفع"))
    reference_number = models.CharField(max_length=100, blank=True, null=True, unique=True, verbose_name=_("الرقم المرجعي"))
    payment_receipt_file = models.FileField(upload_to='payment_receipts/', blank=True, null=True, verbose_name=_("إيصال الدفع"))
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='Pending', verbose_name=_("حالة الدفعة"))
    received_by = models.ForeignKey(FacultyProfile, on_delete=models.SET_NULL, null=True, blank=True,
                                    verbose_name=_("استلمها الموظف"))

    class Meta:
        verbose_name = _("دفعة طالب")
        verbose_name_plural = _("دفعات الطلاب")
        ordering = ['-payment_date']

    def __str__(self):
        return f"دفعة {self.amount} لـ {self.invoice.student.first_name_ar} {self.invoice.student.fourth_name_ar} ({self.invoice.invoice_number}) - {self.status}"