# examination_management/models.py

from django.db import models
from django.utils.translation import gettext_lazy as _
from student_management.models import Student, SemesterEnrollment
from course_management.models import Course, CourseOffering
from university_structure.models import AcademicYear, AcademicSemester, Department
from users.models import FacultyProfile # For grade entry by faculty


class GradeEntry(models.Model):
    """
    Records a student's grade for a specific course offering.
    This is the raw grade entered by the faculty.
    """
    student = models.ForeignKey(Student, on_delete=models.CASCADE, related_name='grades_entered', verbose_name=_("الطالب"))
    course_offering = models.ForeignKey(CourseOffering, on_delete=models.CASCADE, related_name='grades_recorded', verbose_name=_("طرح المقرر"))
    
    # Grade can be numeric (e.g., 85) or letter (e.g., A, B)
    # For simplicity, we'll use numeric, but a comprehensive system might use a GradeScale model.
    score = models.DecimalField(max_digits=5, decimal_places=2, null=True, blank=True, verbose_name=_("الدرجة/النقاط"))
    
    # Final letter grade (e.g., A, B, C, F)
    # This might be calculated based on score and grading scale
    letter_grade = models.CharField(max_length=5, blank=True, null=True, verbose_name=_("التقدير الحرفي"))

    # Faculty who entered the grade
    entered_by = models.ForeignKey(FacultyProfile, on_delete=models.SET_NULL, null=True, blank=True, verbose_name=_("أدخلها الأستاذ"))
    
    entry_date = models.DateTimeField(auto_now_add=True, verbose_name=_("تاريخ الإدخال"))
    
    class Meta:
        unique_together = ('student', 'course_offering') # A student can have only one grade per course offering
        verbose_name = _("إدخال درجة")
        verbose_name_plural = _("إدخالات الدرجات")

    def __str__(self):
        return f"درجة {self.student.first_name_ar} - {self.course_offering.course.code}: {self.score or self.letter_grade or 'N/A'}"


class SemesterResult(models.Model):
    """
    Represents a student's final academic result for a specific semester.
    Includes GPA calculation.
    """
    student = models.ForeignKey(Student, on_delete=models.CASCADE, related_name='semester_results', verbose_name=_("الطالب"))
    academic_year = models.ForeignKey(AcademicYear, on_delete=models.CASCADE, verbose_name=_("السنة الأكاديمية"))
    academic_semester = models.ForeignKey(AcademicSemester, on_delete=models.CASCADE, verbose_name=_("الفصل الدراسي"))
    
    gpa = models.DecimalField(max_digits=4, decimal_places=2, null=True, blank=True, verbose_name=_("المعدل الفصلي (GPA)"))
    
    # This status reflects the approval workflow
    STATUS_CHOICES = [
        ('Pending', _('قيد المراجعة')),
        ('HOD Approved', _('اعتماد رئيس القسم')),
        ('Committee Approved', _('اعتماد لجنة الجامعة')),
        ('Published', _('منشورة')),
        ('Archived', _('مؤرشفة')),
    ]
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='Pending', verbose_name=_("حالة النتيجة"))
    
    approved_by_hod = models.ForeignKey(FacultyProfile, on_delete=models.SET_NULL, null=True, blank=True,
                                        related_name='results_approved_as_hod', verbose_name=_("اعتماد رئيس القسم"))
    approved_by_committee_member = models.ForeignKey(FacultyProfile, on_delete=models.SET_NULL, null=True, blank=True,
                                                    related_name='results_approved_as_committee', verbose_name=_("اعتماد لجنة الجامعة"))
    
    publication_date = models.DateTimeField(null=True, blank=True, verbose_name=_("تاريخ النشر"))
    is_locked = models.BooleanField(default=False, verbose_name=_("مقفلة (غير قابلة للتعديل)"))

    class Meta:
        unique_together = ('student', 'academic_semester') # A student has one result per semester
        verbose_name = _("نتيجة فصلية")
        verbose_name_plural = _("النتائج الفصلية")
        ordering = ['-academic_year__year', '-academic_semester__semester_number', 'student__fourth_name_ar']

    def __str__(self):
        return f"نتيجة {self.student.first_name_ar} {self.student.fourth_name_ar} - {self.academic_semester.name_ar}: GPA {self.gpa or 'N/A'}"


class OverallResult(models.Model):
    """
    Represents a student's cumulative academic result.
    Includes CGPA calculation.
    """
    student = models.OneToOneField(Student, on_delete=models.CASCADE, related_name='overall_result', verbose_name=_("الطالب"))
    cgpa = models.DecimalField(max_digits=4, decimal_places=2, null=True, blank=True, verbose_name=_("المعدل التراكمي (CGPA)"))
    
    academic_standing = models.CharField(max_length=50, blank=True, null=True, verbose_name=_("الوضع الأكاديمي")) # e.g., 'Good Standing', 'Probation', 'Suspended'
    
    # Track progress towards graduation
    total_credits_earned = models.DecimalField(max_digits=5, decimal_places=1, default=0.0, verbose_name=_("إجمالي الساعات المعتمدة المكتسبة"))
    is_graduated = models.BooleanField(default=False, verbose_name=_("متخرج"))
    graduation_date = models.DateField(null=True, blank=True, verbose_name=_("تاريخ التخرج"))

    class Meta:
        verbose_name = _("نتيجة شاملة")
        verbose_name_plural = _("النتائج الشاملة")

    def __str__(self):
        return f"النتيجة الشاملة لـ {self.student.first_name_ar} {self.student.fourth_name_ar}: CGPA {self.cgpa or 'N/A'}"


class StudentCourseResult(models.Model):
    """
    A definitive record of a student's final result in a specific course.
    This model is crucial for tracking failed courses for re-registration.
    """
    student = models.ForeignKey(Student, on_delete=models.CASCADE, related_name='course_results', verbose_name=_("الطالب"))
    course = models.ForeignKey(Course, on_delete=models.CASCADE, verbose_name=_("المقرر الدراسي"))
    academic_semester = models.ForeignKey(AcademicSemester, on_delete=models.CASCADE, verbose_name=_("الفصل الدراسي"))
    
    # Final confirmed grade for this course (after all calculations/approvals)
    final_score = models.DecimalField(max_digits=5, decimal_places=2, null=True, blank=True, verbose_name=_("الدرجة النهائية"))
    final_letter_grade = models.CharField(max_length=5, blank=True, null=True, verbose_name=_("التقدير النهائي"))
    
    # Pass/Fail status - IMPORTANT for re-registration logic
    PASSED_CHOICES = [
        ('Pass', _('ناجح')),
        ('Fail', _('راسب')),
        ('Incomplete', _('غير مكتمل')),
    ]
    status = models.CharField(max_length=10, choices=PASSED_CHOICES, verbose_name=_("الحالة في المقرر"))
    
    # Link to the SemesterResult this course result contributes to
    semester_result = models.ForeignKey(SemesterResult, on_delete=models.SET_NULL, null=True, blank=True,
                                        related_name='individual_course_results', verbose_name=_("النتيجة الفصلية"))

    class Meta:
        unique_together = ('student', 'course', 'academic_semester') # A student has one result for a course per semester
        verbose_name = _("نتيجة مقرر الطالب")
        verbose_name_plural = _("نتائج مقررات الطلاب")
        ordering = ['student', 'academic_semester', 'course__code']

    def __str__(self):
        return f"{self.student.fourth_name_ar} - {self.course.code}: {self.final_letter_grade or self.status}"