# course_management/models.py

from django.db import models
from django.utils.translation import gettext_lazy as _
from university_structure.models import Department, AcademicYear, AcademicSemester
from users.models import FacultyProfile # For assigning faculty to courses

class Course(models.Model):
    """
    Represents a single course offered by the university.
    All courses are considered compulsory for their assigned academic year/semester.
    """
    code = models.CharField(max_length=20, unique=True, verbose_name=_("كود المقرر"))
    title_ar = models.CharField(max_length=255, verbose_name=_("عنوان المقرر (عربي)"))
    title_en = models.CharField(max_length=255, verbose_name=_("Course Title (English)"))
    description_ar = models.TextField(blank=True, verbose_name=_("الوصف (عربي)"))
    # description_en field has been removed as per your request.
    credits = models.DecimalField(max_digits=3, decimal_places=1, verbose_name=_("عدد الساعات المعتمدة"))
    department = models.ForeignKey(Department, on_delete=models.SET_NULL, null=True, blank=True,
                                   related_name='courses', verbose_name=_("القسم المسؤول"))
    
    # Prerequisite courses (Many-to-Many relationship with self)
    # This implies that even compulsory courses might have prerequisites.
    prerequisites = models.ManyToManyField('self', blank=True, symmetrical=False,
                                         related_name='required_for', verbose_name=_("المتطلبات السابقة"))

    class Meta:
        verbose_name = _("مقرر دراسي")
        verbose_name_plural = _("المقررات الدراسية")

    def __str__(self):
        return f"{self.code} - {self.title_ar}"


class ProgramCourse(models.Model):
    """
    Defines the compulsory courses for a specific department at a given academic level/year and semester.
    This replaces the 'Curriculum' model to be more explicit about compulsory courses.
    """
    department = models.ForeignKey(Department, on_delete=models.CASCADE, related_name='program_courses', verbose_name=_("القسم"))
    course = models.ForeignKey(Course, on_delete=models.CASCADE, related_name='program_definitions', verbose_name=_("المقرر الدراسي"))
    
    # The academic year/level this course is compulsory for
    academic_year_level = models.IntegerField(verbose_name=_("المستوى الدراسي/السنة الأكاديمية")) # e.g., 1, 2, 3, 4
    
    # The semester this course is offered in (within that academic year/level)
    semester_number = models.IntegerField(verbose_name=_("رقم الفصل الدراسي")) # e.g., 1 (Fall), 2 (Spring)

    class Meta:
        # A course is compulsory for a specific department at a specific academic level and semester.
        unique_together = ('department', 'course', 'academic_year_level', 'semester_number')
        verbose_name = _("مقرر ضمن الخطة")
        verbose_name_plural = _("مقررات ضمن الخطة")
        ordering = ['department', 'academic_year_level', 'semester_number', 'course__code']


    def __str__(self):
        return f"{self.department.name_ar} - سنة {self.academic_year_level} - فصل {self.semester_number}: {self.course.code}"


class CourseOffering(models.Model):
    """
    Represents an instance of a course being offered in a specific semester.
    This is what students actually get automatically registered into or manually re-register for.
    """
    course = models.ForeignKey(Course, on_delete=models.CASCADE, related_name='offerings', verbose_name=_("المقرر الدراسي"))
    academic_semester = models.ForeignKey(AcademicSemester, on_delete=models.CASCADE, related_name='course_offerings', verbose_name=_("الفصل الدراسي"))
    
    # Faculty member assigned to teach this offering
    faculty = models.ForeignKey(FacultyProfile, on_delete=models.SET_NULL, null=True, blank=True,
                                related_name='course_assignments', verbose_name=_("أستاذ المقرر"))
    
    section_name = models.CharField(max_length=50, blank=True, verbose_name=_("اسم الشعبة (إن وجد)")) # e.g., "Section A", "Lab 1"
    
    # Basic scheduling details
    start_time = models.TimeField(null=True, blank=True, verbose_name=_("وقت البدء"))
    end_time = models.TimeField(null=True, blank=True, verbose_name=_("وقت الانتهاء"))
    days_of_week = models.CharField(max_length=50, blank=True, verbose_name=_("أيام الأسبوع (مثال: MWF)")) # e.g., "MWF", "TTh"
    room_number = models.CharField(max_length=50, blank=True, verbose_name=_("رقم القاعة/المعمل"))

    capacity = models.PositiveIntegerField(default=0, verbose_name=_("السعة القصوى للطلاب"))
    
    class Meta:
        unique_together = ('course', 'academic_semester', 'section_name') # A course can have multiple sections per semester
        verbose_name = _("طرح مقرر")
        verbose_name_plural = _("طروحات المقررات")

    def __str__(self):
        return f"{self.course.code} - {self.academic_semester.name_ar} ({self.section_name or _('عام')})"